// (C) Copyright 2015 Martin Dougiamas
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

angular.module('mm.addons.mod_iomadcertificate')

/**
 * IomadCertificate service.
 *
 * @module mm.addons.mod_iomadcertificate
 * @ngdoc service
 * @name $mmaModIomadCertificate
 */
.factory('$mmaModIomadCertificate', function($q, $mmSite, $mmFS, $mmUtil, $mmSitesManager, mmaModIomadCertificateComponent, $mmFilepool) {
    var self = {};

    /**
     * Get a IomadCertificate.
     *
     * @module mm.addons.mod_iomadcertificate
     * @ngdoc method
     * @name $mmaModIomadCertificate#getIomadCertificate
     * @param {Number} courseId Course ID.
     * @param {Number} cmId     Course module ID.
     * @return {Promise}        Promise resolved when the IomadCertificate is retrieved.
     */
    self.getIomadCertificate = function(courseId, cmId) {
        var params = {
                courseids: [courseId]
            },
            preSets = {
                cacheKey: getIomadCertificateCacheKey(courseId)
            };

        return $mmSite.read('mod_iomadcertificate_get_iomadcertificates_by_courses', params, preSets).then(function(response) {
            if (response.iomadcertificates) {
                var currentIomadCertificate;
                angular.forEach(response.iomadcertificates, function(iomadcertificate) {
                    if (iomadcertificate.coursemodule == cmId) {
                        currentIomadCertificate = iomadcertificate;
                    }
                });
                if (currentIomadCertificate) {
                    return currentIomadCertificate;
                }
            }
            return $q.reject();
        });
    };

    /**
     * Get cache key for IomadCertificate data WS calls.
     *
     * @param {Number} courseId Course ID.
     * @return {String}         Cache key.
     */
    function getIomadCertificateCacheKey(courseId) {
        return 'mmaModIomadCertificate:iomadcertificate:' + courseId;
    }

    /**
     * Get issued iomadcertificates.
     *
     * @module mm.addons.mod_iomadcertificate
     * @ngdoc method
     * @name $mmaModIomadCertificate#getIssuedIomadCertificates
     * @param {Number} id IomadCertificate ID.
     * @return {Promise}  Promise resolved when the issued data is retrieved.
     */
    self.getIssuedIomadCertificates = function(id) {
        var params = {
                iomadcertificateid: id
            },
            preSets = {
                cacheKey: getIssuedIomadCertificatesCacheKey(id)
            };

        return $mmSite.read('mod_iomadcertificate_get_issued_iomadcertificates', params, preSets).then(function(response) {
            if (response.issues) {
                return response.issues;
            }
            return $q.reject();
        });
    };

    /**
     * Get cache key for IomadCertificate issued data WS calls.
     *
     * @param {Number} id IomadCertificate ID.
     * @return {String}   Cache key.
     */
    function getIssuedIomadCertificatesCacheKey(id) {
        return 'mmaModIomadCertificate:issued:' + id;
    }

    /**
     * Invalidates IomadCertificate data.
     *
     * @module mm.addons.mod_iomadcertificate
     * @ngdoc method
     * @name $mmaModIomadCertificate#invalidateIomadCertificate
     * @param {Number} courseId Course ID.
     * @return {Promise}        Promise resolved when the data is invalidated.
     */
    self.invalidateIomadCertificate = function(courseId) {
        return $mmSite.invalidateWsCacheForKey(getIomadCertificateCacheKey(courseId));
    };

    /**
     * Invalidates issues iomadcertificates.
     *
     * @module mm.addons.mod_iomadcertificate
     * @ngdoc method
     * @name $mmaModIomadCertificate#invalidateIssuedIomadCertificates
     * @param {Number} id IomadCertificate ID.
     * @return {Promise}  Promise resolved when the data is invalidated.
     */
    self.invalidateIssuedIomadCertificates = function(id) {
        return $mmSite.invalidateWsCacheForKey(getIssuedIomadCertificatesCacheKey(id));
    };

    /**
     * Return whether or not the plugin is enabled in a certain site. Plugin is enabled if the iomadcertificate WS are available.
     *
     * @module mm.addons.mod_iomadcertificate
     * @ngdoc method
     * @name $mmaModIomadCertificate#isPluginEnabled
     * @param  {String} [siteId] Site ID. If not defined, current site.
     * @return {Promise}         Promise resolved with true if plugin is enabled, rejected or resolved with false otherwise.
     */
    self.isPluginEnabled = function(siteId) {
        siteId = siteId || $mmSite.getId();

        return $mmSitesManager.getSite(siteId).then(function(site) {
            return site.wsAvailable('mod_iomadcertificate_get_iomadcertificates_by_courses');
        });
    };

    /**
     * Report the IomadCertificate as being viewed.
     *
     * @module mm.addons.mod_iomadcertificate
     * @ngdoc method
     * @name $mmaModIomadCertificate#logView
     * @param {String} id IomadCertificate ID.
     * @return {Promise}  Promise resolved when the WS call is successful.
     */
    self.logView = function(id) {
        if (id) {
            var params = {
                iomadcertificateid: id
            };
            return $mmSite.write('mod_iomadcertificate_view_iomadcertificate', params);
        }
        return $q.reject();
    };

    /**
     * Issue a iomadcertificate.
     *
     * @module mm.addons.mod_iomadcertificate
     * @ngdoc method
     * @name $mmaModIomadCertificate#issueIomadCertificate
     * @param {Number} iomadcertificateId IomadCertificate ID.
     * @return {Promise}  Promise resolved when the WS call is successful.
     */
    self.issueIomadCertificate = function(iomadcertificateId) {
         var params = {
            iomadcertificateid: iomadcertificateId
        };
        return $mmSite.write('mod_iomadcertificate_issue_iomadcertificate', params).then(function(response) {
            if (!response || !response.issue) {
                return $q.reject();
            }
        });
    };

    /**
     * Download or open a downloaded iomadcertificate.
     *
     * @module mm.addons.mod_iomadcertificate
     * @ngdoc method
     * @name $mmaModIomadCertificate#openIomadCertificate
     * @param {Object} issuedIomadCertificate Issued iomadcertificate object.
     * @param {Number} moduleId Module id.
     * @return {Promise}  Promise resolved when the WS call is successful.
     */
    self.openIomadCertificate = function(issuedIomadCertificate, moduleId) {

        var siteId = $mmSite.getId(),
            revision = 0,
            timeMod = issuedIomadCertificate.timecreated,
            files = [{fileurl: issuedIomadCertificate.fileurl, filename: issuedIomadCertificate.filename, timemodified: timeMod}];
        if ($mmFS.isAvailable()) {
            // The file system is available.
            promise = $mmFilepool.downloadPackage(siteId, files, mmaModIomadCertificateComponent, moduleId, revision, timeMod).then(function() {
                return $mmFilepool.getUrlByUrl(siteId, issuedIomadCertificate.fileurl, mmaModIomadCertificateComponent, moduleId, timeMod);
            });
        } else {
            // We use the live URL.
            promise = $q.when($mmSite.fixPluginfileURL(issuedIomadCertificate.fileurl));
        }

        return promise.then(function(localUrl) {
            return $mmUtil.openFile(localUrl);
        });
    };

    /**
     * Invalidate downloaded iomadcertificates.
     *
     * @module mm.addons.mod_iomadcertificate
     * @ngdoc method
     * @name $mmaModIomadCertificate#invalidateDownloadedIomadCertificates
     * @param {Number} moduleId Module id.
     * @return {Promise}  Promise resolved when the WS call is successful.
     */
    self.invalidateDownloadedIomadCertificates = function(moduleId) {
        return $mmFilepool.invalidateFilesByComponent($mmSite.getId(), mmaModIomadCertificateComponent, moduleId);
    };

    return self;
});
